#include "hnfsflt.h"
#include "rfs.h"

redirfs_filter hnfsflt;

/*
 * Gets the dentry inode number.
 */
static unsigned long hnfsflt_get_ino(struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	if (!inode) {
		/* No inode either, fall back to 0 (dirty) */
		return 0;
	} else {
		return inode->i_ino;
	}
}

/*
 * Gets the dentry dev number.
 */
static unsigned long hnfsflt_get_dev(struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	if (!inode) {
		/* No inode either, fall back to 0 (dirty) */
		return 0;
	} else {
		struct super_block *sb = inode->i_sb;
		if (!sb) {
			/* No sb (should not happen) */
			return 0;
		} else {
			return huge_encode_dev(sb->s_dev);
		}
	}
}

#define DELETED_SUFFIX	 " (deleted)"
#define DELETED_SUFFIX_LEN 10

/*
 * Gets the filename from a dentry. If the dentry is not in filter path,
 * we return the inode.
 * May return NULL, i.e. -ENOMEM.
 *
 * @removeDeleted if true, remove the suffix ' (deleted)' if present
 */

static char *hnfsflt_get_path(struct dentry *dentry, struct vfsmount *mnt, int removeDeleted)
{
	int ret = 0;
	unsigned long size_of_full_path = 2 * PATH_MAX;
	char* full_path = NULL;
	redirfs_root root = NULL;
	redirfs_path *paths_flt = NULL;
	struct redirfs_path_info *path_flt_info = NULL;
	int mnt_rfs_lookup = 0;
	if (mnt == NULL) {
		mnt_rfs_lookup = 1;
		root = redirfs_get_root_dentry(hnfsflt, dentry);
		if (!root) {
			/* Might happen if path is outside filter root.
				Best thing we can do here is return the inode */
			unsigned long ino;

			if (!dentry->d_inode) {
				/* No inode either, fall back to 0 (dirty) */
				ino = 0;
			} else {
				ino = dentry->d_inode->i_ino;
			}

			full_path = kzalloc(sizeof(char) * size_of_full_path, GFP_KERNEL);
			if (!full_path) {
				printk(KERN_WARNING "hnfsflt: full_path = kzalloc failed\n");
				return NULL;
			}

			snprintf(full_path, size_of_full_path, "%lu", ino);
			goto fail_root;
		}

		paths_flt = redirfs_get_paths_root(hnfsflt, root);
		if (IS_ERR(paths_flt) || !paths_flt) {
			printk(KERN_WARNING "hnfsflt: redirfs_get_paths failed\n");
			goto fail_paths_flt;
		}

		path_flt_info = redirfs_get_path_info(hnfsflt, paths_flt[0]);
		if (IS_ERR(path_flt_info)) {
			printk(KERN_WARNING "hnfsflt: redirfs_get_path_info failed\n");
			goto fail_path_info;
		}
		mnt = path_flt_info->mnt;
	}

	/* The filename is at most PAGE_SIZE long, but there may have	*/
	/* some extra chars: ' (deleted)' or other extension		*/
	full_path = kzalloc(size_of_full_path, GFP_KERNEL);
	if (!full_path) {
		printk(KERN_WARNING "hnfsflt: filename = kzalloc failed\n");
		goto fail_filename_alloc;
	}


	ret = redirfs_get_filename(mnt, dentry, full_path, size_of_full_path);
	if (ret) {
		printk(KERN_ERR "hnfsflt: redirfs_get_filename failed\n");
		kfree(full_path);
		full_path = NULL;
	}

	/* Remove suffix ' (deleted)' if present */
	if (removeDeleted && full_path != NULL) {
		size_t filelen = strlen(full_path);
		size_t sufflen = DELETED_SUFFIX_LEN;
		if (filelen > sufflen) {
			char *suffix = full_path + filelen - sufflen;
			if (strcmp(suffix, DELETED_SUFFIX) == 0) {
				*suffix = '\0';
			}
		}
	}

fail_filename_alloc:
	if (mnt_rfs_lookup == 1) redirfs_put_path_info(path_flt_info);
fail_path_info:
	if (mnt_rfs_lookup == 1) redirfs_put_paths(paths_flt);
fail_paths_flt:
	if (mnt_rfs_lookup == 1) redirfs_put_root(root);
fail_root:
	return full_path;
}

static char hnfsflt_get_filetype(struct dentry *dentry)
{
	umode_t i_mode;

	if (!dentry || !dentry->d_inode)
		return 'U';

	i_mode = dentry->d_inode->i_mode;
	if (S_ISREG(i_mode))
		return 'R';
	else if (S_ISLNK(i_mode))
		return 'L';
	else if (S_ISDIR(i_mode))
		return 'D';

	return 'U';
}


/*
 * Create the logline for the given request.
 * return the new logline, NULL if the event is ignored or an ERR_PTR.
 */
static struct hnfsflt_logline *hnfsflt_create_ll(
	redirfs_context context, struct redirfs_args *args)
{
	struct hnfsflt_logline* logline = NULL;
	char* path = NULL;
	struct file* file = NULL;
	struct vfsmount* vfsmnt = NULL;
	struct vfsmount* mnt = NULL;

	char hnfsflt_type = 0;
	char hnfsflt_op = 0;
	unsigned long ino = 0;
	unsigned long dev = 0;
	int writers = 0;

	/* Fill the logline depending on the call */
	switch (args->type.id) {
		case REDIRFS_REG_FOP_RELEASE:
			/* Log last writer release */
			if (!(args->args.f_release.file->f_mode & FMODE_WRITE)) {
				/* Not opened for write (ignored) */
				return NULL;
			}
			writers = atomic_read(&args->args.f_release.inode->i_writecount);
			if (writers != 1) {
				return NULL;
			}
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,19,0))
            ino = hnfsflt_get_ino(args->args.f_release.file->f_dentry);
            dev = hnfsflt_get_dev(args->args.f_release.file->f_dentry);
#else
            ino = hnfsflt_get_ino(args->args.f_release.file->f_path.dentry);
            dev = hnfsflt_get_dev(args->args.f_release.file->f_path.dentry);
#endif

			file = args->args.f_release.file;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,9,0))
			vfsmnt = file->f_vfsmnt;
#else
			vfsmnt = file->f_path.mnt;
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,19,0))
            path = hnfsflt_get_path(args->args.f_release.file->f_dentry, vfsmnt, 0);
#else
            path = hnfsflt_get_path(args->args.f_release.file->f_path.dentry, vfsmnt, 0);
#endif
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'R';
			hnfsflt_op = 'W';
			break;

		case REDIRFS_REG_IOP_SETATTR:
			ino = hnfsflt_get_ino(args->args.i_setattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_setattr.dentry);
			path = hnfsflt_get_path(args->args.i_setattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'R';
			hnfsflt_op = 'A';
			break;

		case REDIRFS_REG_IOP_SETXATTR:
			ino = hnfsflt_get_ino(args->args.i_setxattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_setxattr.dentry);
			path = hnfsflt_get_path(args->args.i_setxattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'R';
			hnfsflt_op = 'X';
			break;

		case REDIRFS_REG_IOP_REMOVEXATTR:
			ino = hnfsflt_get_ino(args->args.i_removexattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_removexattr.dentry);
			path = hnfsflt_get_path(args->args.i_removexattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'R';
			hnfsflt_op = 'Y';
			break;

		case REDIRFS_DIR_IOP_CREATE:
			ino = hnfsflt_get_ino(args->args.i_create.dentry);
			dev = hnfsflt_get_dev(args->args.i_create.dentry);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,6,0))
			mnt = (args->args.i_create.nd == NULL ? NULL : rfs_nameidata_mnt(args->args.i_create.nd));
#endif
			path = hnfsflt_get_path(args->args.i_create.dentry, mnt, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'R';
			hnfsflt_op = 'C';
			break;

		case REDIRFS_DIR_IOP_LINK:
			ino = hnfsflt_get_ino(args->args.i_link.dentry);
			dev = hnfsflt_get_dev(args->args.i_link.dentry);
			path = hnfsflt_get_path(args->args.i_link.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = hnfsflt_get_filetype(args->args.i_link.old_dentry);
			hnfsflt_op = 'C';
			break;

		case REDIRFS_DIR_IOP_UNLINK:
			ino = hnfsflt_get_ino(args->args.i_unlink.dentry);
			dev = hnfsflt_get_dev(args->args.i_unlink.dentry);
			path = hnfsflt_get_path(args->args.i_unlink.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = hnfsflt_get_filetype(args->args.i_unlink.dentry);
			hnfsflt_op = 'D';
			break;

		case REDIRFS_DIR_IOP_SYMLINK:
			ino = hnfsflt_get_ino(args->args.i_symlink.dentry);
			dev = hnfsflt_get_dev(args->args.i_symlink.dentry);
			path = hnfsflt_get_path(args->args.i_symlink.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'L';
			hnfsflt_op = 'C';
			break;

		case REDIRFS_DIR_IOP_MKDIR:
			ino = hnfsflt_get_ino(args->args.i_mkdir.dentry);
			dev = hnfsflt_get_dev(args->args.i_mkdir.dentry);
			path = hnfsflt_get_path(args->args.i_mkdir.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'D';
			hnfsflt_op = 'C';
			break;

		case REDIRFS_DIR_IOP_RMDIR:
			ino = hnfsflt_get_ino(args->args.i_rmdir.dentry);
			dev = hnfsflt_get_dev(args->args.i_rmdir.dentry);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0))
			path = hnfsflt_get_path(args->args.i_rmdir.dentry, NULL, 1);
#else
			path = hnfsflt_get_path(args->args.i_rmdir.dentry, NULL, 0);
#endif
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'D';
			hnfsflt_op = 'D';
			break;

		case REDIRFS_DIR_IOP_RENAME:
			ino = hnfsflt_get_ino(args->args.i_rename.old_dentry);
			dev = hnfsflt_get_dev(args->args.i_rename.old_dentry);
			path = hnfsflt_get_path(args->args.i_rename.new_dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = hnfsflt_get_filetype(args->args.i_rename.old_dentry);
			hnfsflt_op = 'I';
			break;

		case REDIRFS_DIR_IOP_SETATTR:
			ino = hnfsflt_get_ino(args->args.i_setattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_setattr.dentry);
			path = hnfsflt_get_path(args->args.i_setattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'D';
			hnfsflt_op = 'A';
			break;

		case REDIRFS_DIR_IOP_SETXATTR:
			ino = hnfsflt_get_ino(args->args.i_setxattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_setxattr.dentry);
			path = hnfsflt_get_path(args->args.i_setxattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'D';
			hnfsflt_op = 'X';
			break;

		case REDIRFS_DIR_IOP_REMOVEXATTR:
			ino = hnfsflt_get_ino(args->args.i_removexattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_removexattr.dentry);
			path = hnfsflt_get_path(args->args.i_removexattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'D';
			hnfsflt_op = 'Y';
			break;

		case REDIRFS_LNK_IOP_SETATTR:
			ino = hnfsflt_get_ino(args->args.i_setattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_setattr.dentry);
			path = hnfsflt_get_path(args->args.i_setattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'L';
			hnfsflt_op = 'A';
			break;

		case REDIRFS_LNK_IOP_SETXATTR:
			ino = hnfsflt_get_ino(args->args.i_setxattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_setxattr.dentry);
			path = hnfsflt_get_path(args->args.i_setxattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'L';
			hnfsflt_op = 'X';
			break;

		case REDIRFS_LNK_IOP_REMOVEXATTR:
			ino = hnfsflt_get_ino(args->args.i_removexattr.dentry);
			dev = hnfsflt_get_dev(args->args.i_removexattr.dentry);
			path = hnfsflt_get_path(args->args.i_removexattr.dentry, NULL, 0);
			if (!path) {
				goto fail_getpath;
			}
			hnfsflt_type = 'L';
			hnfsflt_op = 'Y';
			break;

		default:
			ino = 0;
			dev = 0;
			hnfsflt_type = '?';
			hnfsflt_op = '?';
	}

	/* Allocate logline */
	logline = hnfsflt_ll_alloc(hnfsflt_type, hnfsflt_op, ino, dev, path);

	/* Can release path mem */
fail_getpath:
	if (path) {
		kfree(path);
		path = NULL;
	}

	/* logline may be NULL, an ERR_PTR or a logline */
	return logline;
}


/*
 * Nop callback. Seems to go faster when there is a precall callback!
 */
static enum redirfs_rv hnfsflt_callback_nop(
	redirfs_context context, struct redirfs_args *args)
{
	return REDIRFS_CONTINUE;
}


/*
 * Create and add logline to ring buffer.
 * TODO: take command failure into account to ignore operation?
 */
static enum redirfs_rv hnfsflt_logcall(
	redirfs_context context, struct redirfs_args *args)
{
	struct hnfsflt_logline *logline;

	if (args->rv.rv_int < 0) {
		/* Operation failed: ignore silently */
		// printk(KERN_DEBUG "hnfsflt: op failed %d (%d)\n", args->type.id, args->rv.rv_int);
		return REDIRFS_CONTINUE;
	}
	logline = hnfsflt_create_ll(context, args);
	if (logline == NULL) {
		/* Ignore silently */
		return REDIRFS_CONTINUE;
	}
	if (IS_ERR(logline)) {
		printk(KERN_ERR "hnfsflt: create logline failed (%ld)\n", PTR_ERR(logline));
		return REDIRFS_CONTINUE;
	}

	/* Insert into ringbuffer */
	hnfsflt_ll_insert(logline);

	return REDIRFS_CONTINUE;
}

/*
 * MOVE OUT
 * Need to save the path during the pre callback (can not be
 * computed during the post callback directly).
 */

/*
 * NOP, freeing is done by the character device on read or
 * here if the log line is not enqueued.
 */
static void hnfsflt_rfsdata_free(struct redirfs_data *unused)
{
	/* Nothing to do */
}

/*
 * Move OUT pre callback. Compute moved entry OUT path and store
 * the log line in the context.
 */
static enum redirfs_rv hnfsflt_moveout_pre(
	redirfs_context context, struct redirfs_args *args)
{
	int ret;
	char hnfsflt_type, hnfsflt_op;
	unsigned long ino;
	unsigned long dev;
	char *path = NULL;
	struct hnfsflt_logline *logline = NULL;
	struct redirfs_data *rfs_data;

	if (args->type.id != REDIRFS_DIR_IOP_RENAME) {
		BUG();
		/* Not reached */
		return REDIRFS_CONTINUE;
	}

	ino = hnfsflt_get_ino(args->args.i_rename.old_dentry);
	dev = hnfsflt_get_dev(args->args.i_rename.old_dentry);
	path = hnfsflt_get_path(args->args.i_rename.old_dentry, NULL, 0);
	if (!path) {
		return REDIRFS_CONTINUE;
	}
	hnfsflt_type = hnfsflt_get_filetype(args->args.i_rename.old_dentry);
	hnfsflt_op = 'O';

	/* Allocate logline and release path */
	logline = hnfsflt_ll_alloc(hnfsflt_type, hnfsflt_op, ino, dev, path);
	kfree(path);
	if (IS_ERR(logline)) {
		printk(KERN_ERR "hnfsflt: create logline failed (%ld)\n", PTR_ERR(logline));
		return REDIRFS_CONTINUE;
	}

	/* Attach to context */
	ret = redirfs_init_data(&logline->rfs_data, hnfsflt, hnfsflt_rfsdata_free, NULL);
	if (ret) {
		printk(KERN_ERR "hnfsflt: redirfs_init_data() failed for %s (%d)\n", logline->path, ret);
		hnfsflt_ll_free(logline);
		return REDIRFS_CONTINUE;
	}

	rfs_data = redirfs_attach_data_context(hnfsflt, context, &logline->rfs_data);
	if (!rfs_data) {
		printk(KERN_ERR "hnfsflt: redirfs_attach_data_context() failed for %s\n", logline->path);
		hnfsflt_ll_free(logline);
		return REDIRFS_CONTINUE;
	}
	/* rfs_data is now referenced 3 times */
	redirfs_put_data(rfs_data);

	return REDIRFS_CONTINUE;
}


/*
 * Move OUT post callback. On success, create a log line with
 * the path computed in the pre callback.
 */
static enum redirfs_rv hnfsflt_moveout_post(
		redirfs_context context, struct redirfs_args *args)
{
	struct redirfs_data *rfs_data;
	struct hnfsflt_logline *logline;

	/* Get log line from context. Ignore silently if the data have not been added */
	rfs_data = redirfs_detach_data_context(hnfsflt, context);
	if (!rfs_data)
		return REDIRFS_CONTINUE;

	/* Get log line and release data */
	logline = container_of(rfs_data, struct hnfsflt_logline, rfs_data);
	redirfs_put_data(rfs_data);
	redirfs_put_data(rfs_data);

	if (args->rv.rv_int < 0) {
		/* Operation failed: ignore silently */
		// printk(KERN_DEBUG "hnfsflt: op failed %d (%d)\n", args->type.id, args->rv.rv_int);
		hnfsflt_ll_free(logline);
		return REDIRFS_CONTINUE;
	}

	/* Insert into ringbuffer */
	hnfsflt_ll_insert(logline);

		return REDIRFS_CONTINUE;
}


static struct redirfs_op_info hnfsflt_op_info[] = {
	{REDIRFS_REG_IOP_SETATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_REG_IOP_SETXATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_REG_IOP_REMOVEXATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_REG_FOP_RELEASE, hnfsflt_callback_nop, hnfsflt_logcall},

	{REDIRFS_DIR_IOP_CREATE, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_LINK, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_UNLINK, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_SYMLINK, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_MKDIR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_RMDIR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_RENAME, hnfsflt_moveout_pre, hnfsflt_moveout_post},
	{REDIRFS_DIR_IOP_SETATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_SETXATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_DIR_IOP_REMOVEXATTR, hnfsflt_callback_nop, hnfsflt_logcall},

	{REDIRFS_LNK_IOP_SETATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_LNK_IOP_SETXATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_LNK_IOP_REMOVEXATTR, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_LNK_FOP_OPEN, hnfsflt_callback_nop, hnfsflt_logcall},
	{REDIRFS_LNK_FOP_RELEASE, hnfsflt_callback_nop, hnfsflt_logcall},

	{REDIRFS_OP_END, NULL, NULL}
};


static struct redirfs_filter_operations hnfsflt_ops = {
	.pre_rename = hnfsflt_callback_nop,
	.post_rename = hnfsflt_logcall
};

static struct redirfs_filter_info hnfsflt_info = {
	.owner = THIS_MODULE,
	.name = "hnfsflt",
	.priority = 60321,
	.active = 1,
	.ops = &hnfsflt_ops
};


int hnfsflt_rfs_init(void)
{
	int ret;

	hnfsflt = redirfs_register_filter(&hnfsflt_info);
	if (IS_ERR(hnfsflt))
		return PTR_ERR(hnfsflt);

	ret = redirfs_set_operations(hnfsflt, hnfsflt_op_info);
	if (ret) {
		int err;
		err = redirfs_unregister_filter(hnfsflt);
		if (err)
			printk(KERN_ERR "hnfsflt: unregister filter failed (%d)\n", err);
		else
			redirfs_delete_filter(hnfsflt);
		return ret;
	}

	return 0;
}

void hnfsflt_rfs_cleanup(void)
{
	/* Filter is already unregistered, otherwise module ref count is greater than 0 */
	redirfs_delete_filter(hnfsflt);
}
